import os
import unittest
import warnings


from iotas.html_generator_configuration import HtmlGeneratorConfiguration
from iotas.note import Note
from iotas.template_html_generator import TemplateHtmlGenerator, TemplateDetails

warnings.filterwarnings("ignore", "version")


class _TestHtmlGeneratorConfig(HtmlGeneratorConfiguration):

    def get_markdown_tex_support(self) -> bool:
        """Get whether markdown TeX rendering is supported.

        :return: Markdown TeX supported
        :rtype: bool
        """
        return False

    def get_line_length(self) -> int:
        """Get line length.

        :return: Size in pixels
        :rtype: int
        """
        return 800

    def get_markdown_use_monospace_font(self) -> bool:
        """Get whether to use a monospace font for the markdown render.

        :return: Using monospace font
        :rtype: bool
        """
        return False

    def get_use_monospace_font(self) -> bool:
        """Get whether to use a monospace font.

        :return: Using monospace font
        :rtype: bool
        """
        return True

    def get_font_size(self) -> int:
        """Get font size.

        :return: Size
        :rtype: int
        """
        return 14

    def get_high_contrast(self) -> bool:
        """Get whether high contrast output should be used for screen rendering.

        :return: Whether visible
        :rtype: bool
        """
        return False

    def get_markdown_render_monospace_font_ratio(self) -> float:
        """Get the adjustment in size from proportional to fixed width font.

        :return: Ratio
        :rtype: float
        """
        return 1.0

    def get_editor_header_bar_visible_for_window_state(self) -> bool:
        """Get whether the header bar is configured visible for the window maximised state."

        :return: Whether visible
        :rtype: bool
        """
        return False

    def get_toolbar_underlay_padding_height(self) -> int:
        """Fetch the height of padding to account for overlay header bars.

        :return: Height in pixels
        :rtype: int
        """
        return 47


class Test(unittest.TestCase):

    generator: TemplateHtmlGenerator

    def test_generate(self) -> None:
        note = Note(new_note=True)
        note.content = "content"
        note.title = "Test Note Title"

        generator = self.__reset()
        html, _tokens = generator.generate(note, searching=False, export_format=None)
        self.assertTrue(note.content in html)
        self.assertTrue(note.title in html)
        self.assertFalse("katex" in html)
        self.assertFalse("<style>" in html)
        self.assertFalse("scrollTop = " in html)
        self.assertTrue("/mock/data/path" in html)

        generator = self.__reset()
        html, _tokens = generator.generate(
            note, searching=False, export_format=None, scroll_position=100.0
        )
        self.assertTrue("scrollTop = 100.0" in html)

        generator = self.__reset()
        html, _tokens = generator.generate(note, searching=False, export_format="html")
        self.assertFalse("scrollTop = " in html)
        self.assertTrue("font-size: 14pt" in html)
        self.assertTrue("max-width: 800px" in html)
        self.assertTrue("padding-top: 47px" in html)
        self.assertFalse("::selection {" in html)
        self.assertFalse("/mock/data/path" in html)

    def test_generate_user_stylesheet(self) -> None:
        generator = self.__reset()
        css = generator.generate_user_stylesheet(searching=False)
        self.assertTrue("font-size: 14pt" in css)
        self.assertTrue("max-width: 800px" in css)
        self.assertTrue("padding-top: 47px" in css)
        self.assertFalse("::selection {" in css)

        css = generator.generate_user_stylesheet(searching=True)
        self.assertTrue("::selection {" in css)

    def test_update_font_family(self) -> None:
        generator = self.__reset()
        generator.update_font_family("unique")
        css = generator.generate_user_stylesheet(searching=False)
        self.assertTrue("font-family: unique" in css)

    def __reset(self) -> TemplateHtmlGenerator:
        config = _TestHtmlGeneratorConfig()
        data_path = "/mock/data/path"
        self.generator = TemplateHtmlGenerator(config, data_path, self.__template_loader)
        return self.generator

    def __template_loader(self) -> TemplateDetails:
        file_dir = os.path.dirname(__file__)
        templates_path = os.path.join(file_dir, os.pardir, "data", "render-template")
        with open(os.path.join(templates_path, "template.html"), "r") as f:
            html = f.read()
        with open(os.path.join(templates_path, "binding.js"), "r") as f:
            js = f.read()
        with open(os.path.join(templates_path, "configurable.css"), "r") as f:
            css = f.read()
        with open(os.path.join(templates_path, "searching.css"), "r") as f:
            search_css = f.read()
        return TemplateDetails(html, js, css, search_css)
