// Copyright 2008 Dolphin Emulator Project
// SPDX-License-Identifier: GPL-2.0-or-later
//
// Copyright (C) Hector Martin "marcan" (hector@marcansoft.com)

#include "Core/HW/WiimoteEmu/Encryption.h"

#include <algorithm>
#include <bit>

#include "Common/Logging/Log.h"
namespace
{
using WiimoteEmu::SBox;

constexpr SBox fallback_sbox_1st_party_a = {
    0x00, 0x3b, 0x4c, 0x89, 0x7b, 0xa5, 0x77, 0x17, 0xff, 0x9e, 0x9b, 0x9c, 0x4d, 0xa3, 0xe1, 0x0c,
    0xf2, 0x47, 0x6a, 0xee, 0x3c, 0x8d, 0x46, 0xf1, 0x36, 0x4b, 0x6c, 0xfa, 0xf5, 0xe7, 0x93, 0x12,
    0x81, 0xfc, 0x1f, 0xd9, 0xf0, 0xe9, 0x50, 0xc6, 0xc3, 0x04, 0xf9, 0x2b, 0x71, 0x44, 0xd6, 0xe4,
    0x1b, 0x2d, 0x7a, 0x90, 0xd4, 0x1c, 0x72, 0x1e, 0x0b, 0xe8, 0xec, 0x14, 0xb1, 0x85, 0xd7, 0x8c,
    0x39, 0x5e, 0x32, 0xe0, 0x22, 0x54, 0x68, 0xfb, 0x8a, 0x4f, 0xf7, 0xa6, 0x25, 0xea, 0x7c, 0x61,
    0xbf, 0x99, 0x64, 0xad, 0xc7, 0x57, 0x48, 0xc9, 0x73, 0xbc, 0xb9, 0x0f, 0x9d, 0xcb, 0x35, 0xbd,
    0xdc, 0x3d, 0x03, 0x33, 0x45, 0xce, 0xc0, 0x78, 0xbe, 0xa7, 0x96, 0x5a, 0x83, 0xd0, 0x5c, 0x70,
    0x10, 0xae, 0xaa, 0xf3, 0x63, 0x75, 0x09, 0x40, 0xcd, 0x9f, 0xdf, 0x15, 0xe6, 0xbb, 0xc5, 0x3f,
    0xeb, 0x74, 0x52, 0x3a, 0x18, 0xc8, 0x2e, 0xb3, 0x62, 0x2f, 0xc1, 0x65, 0x67, 0xfe, 0xd1, 0x24,
    0x34, 0xcf, 0xb6, 0x58, 0xa2, 0xef, 0xca, 0x06, 0x98, 0x88, 0x28, 0x43, 0xb4, 0xb5, 0x27, 0x5f,
    0x8e, 0xf4, 0x80, 0xb8, 0x07, 0xa8, 0x91, 0x19, 0x4e, 0xa9, 0x05, 0xab, 0x42, 0x0d, 0x49, 0x13,
    0xb7, 0x6d, 0x87, 0xe2, 0x01, 0x51, 0x41, 0x4a, 0xda, 0x8f, 0xd3, 0x0e, 0x7f, 0xd5, 0x94, 0xdd,
    0x2a, 0x55, 0x92, 0x0a, 0xfd, 0x66, 0xe3, 0xc4, 0x31, 0xb0, 0x60, 0x53, 0x2c, 0x59, 0x26, 0x21,
    0x38, 0xcc, 0x37, 0x29, 0x95, 0x1d, 0x16, 0x84, 0xf8, 0x9a, 0x7d, 0x6f, 0x56, 0xd2, 0xb2, 0x79,
    0x86, 0x6e, 0x3e, 0x1a, 0x7e, 0xed, 0x69, 0x08, 0xba, 0x5b, 0xf6, 0xa0, 0x11, 0x23, 0x02, 0x30,
    0xc2, 0xe5, 0xac, 0x82, 0xd8, 0x97, 0x8b, 0x20, 0x6b, 0xa1, 0xde, 0xdb, 0xaf, 0x5d, 0x76, 0xa4,
};

constexpr SBox fallback_sbox_1st_party_b = {
    0x00, 0xa1, 0xa8, 0x63, 0xd7, 0x3e, 0x84, 0xa6, 0xb7, 0xd5, 0xfb, 0x14, 0x67, 0x16, 0x08, 0xbc,
    0x5c, 0x15, 0x35, 0x27, 0x58, 0x73, 0x90, 0x55, 0x07, 0x4e, 0xf9, 0xb1, 0x5a, 0x75, 0x92, 0x98,
    0x8d, 0xf3, 0x44, 0xcc, 0xeb, 0x4f, 0xac, 0x11, 0x4b, 0xe4, 0xcb, 0xef, 0xde, 0xc7, 0x6e, 0x9e,
    0x89, 0x8f, 0x03, 0xcd, 0x09, 0xa9, 0x76, 0x95, 0x6c, 0x20, 0xb0, 0x29, 0xe5, 0x38, 0x78, 0x97,
    0x61, 0x70, 0x80, 0x17, 0x2f, 0xe7, 0x79, 0xb8, 0xc5, 0x47, 0x9b, 0x43, 0xaf, 0xb6, 0x7d, 0x42,
    0xb2, 0x23, 0x1b, 0x87, 0xc3, 0x33, 0x3c, 0x2c, 0x9d, 0xd3, 0x28, 0xe8, 0x62, 0x9a, 0xd0, 0x30,
    0x39, 0x5f, 0x1f, 0x37, 0x40, 0xba, 0x02, 0x19, 0x2a, 0x3f, 0xbe, 0x69, 0x60, 0xfd, 0x53, 0xc1,
    0xdf, 0xe1, 0x0b, 0x59, 0x12, 0x5b, 0x01, 0xbf, 0x1d, 0x91, 0x0f, 0x52, 0x13, 0xfc, 0xe3, 0x6f,
    0xbb, 0xcf, 0x25, 0x26, 0x45, 0x7e, 0x86, 0xa2, 0xa0, 0xd4, 0x51, 0x41, 0xe2, 0xf8, 0x82, 0xf6,
    0xc6, 0xa3, 0x34, 0xc9, 0xda, 0x18, 0xaa, 0x2e, 0x10, 0x24, 0xec, 0x32, 0x9f, 0x54, 0xe0, 0x49,
    0xae, 0x72, 0x85, 0xdb, 0x2b, 0xab, 0x50, 0xea, 0x9c, 0x94, 0xb3, 0xca, 0xe6, 0x71, 0x81, 0xff,
    0x4d, 0x64, 0x05, 0xee, 0xc4, 0xf0, 0xc2, 0x3b, 0xb5, 0xf4, 0x5e, 0x22, 0x88, 0xdc, 0x31, 0xa4,
    0x8a, 0xd2, 0xf7, 0xdd, 0x4c, 0x65, 0xd6, 0xf1, 0x8e, 0xed, 0x57, 0x36, 0x5d, 0xa5, 0x0c, 0x06,
    0x77, 0x8b, 0x2d, 0x0a, 0xb4, 0xd9, 0xc0, 0x1e, 0xe9, 0x3a, 0xf5, 0x4a, 0x1a, 0x46, 0x6d, 0x48,
    0x66, 0x7a, 0x93, 0xce, 0x74, 0x7f, 0x21, 0xd8, 0x7c, 0x3d, 0x96, 0x7b, 0xd1, 0x04, 0x6a, 0x0e,
    0x1c, 0xfa, 0x83, 0x99, 0x56, 0x8c, 0xf2, 0x6b, 0xbd, 0xad, 0xc8, 0xa7, 0xfe, 0xb9, 0x68, 0x0d,
};

constexpr SBox fallback_sbox_3rd_party_a = {
    0x00, 0xd8, 0x64, 0x01, 0x32, 0x7c, 0x2a, 0x1c, 0x84, 0xcc, 0xde, 0xee, 0xbe, 0xb2, 0x1a, 0xaf,
    0xd7, 0xa8, 0xc6, 0xb0, 0x11, 0x8a, 0x81, 0xbc, 0xd2, 0x50, 0x4c, 0x6a, 0x85, 0x83, 0xb3, 0x07,
    0x30, 0x31, 0x91, 0x5f, 0xf6, 0x26, 0x58, 0xbf, 0xc8, 0x54, 0x62, 0x6b, 0xba, 0xa0, 0x2d, 0x5d,
    0x88, 0x75, 0xb8, 0x36, 0x41, 0x06, 0x51, 0x16, 0x15, 0x52, 0xdd, 0x3e, 0x87, 0xf8, 0x90, 0x13,
    0x6d, 0xb4, 0xe9, 0x69, 0x99, 0x70, 0xf4, 0x25, 0x57, 0xc4, 0x2e, 0xcf, 0xa4, 0xd9, 0x22, 0x68,
    0xae, 0xb9, 0xf2, 0x61, 0xca, 0xf0, 0x1f, 0x8c, 0x5e, 0x78, 0x27, 0x4f, 0x3a, 0x72, 0xff, 0x21,
    0x4a, 0x0a, 0x2f, 0x8b, 0x12, 0x6e, 0x46, 0x96, 0x97, 0xe3, 0x18, 0xa9, 0xcb, 0xce, 0xf5, 0x60,
    0xc5, 0x5b, 0xf3, 0x0f, 0x43, 0xa3, 0x0d, 0x1d, 0xb5, 0x6c, 0xb1, 0xe1, 0xcd, 0x04, 0xdc, 0x7f,
    0x8e, 0xbd, 0xd3, 0x05, 0x47, 0x98, 0x1e, 0xb6, 0xed, 0x73, 0x08, 0x0e, 0x39, 0xfc, 0x3f, 0xef,
    0xa6, 0x3b, 0x93, 0xaa, 0x2c, 0x95, 0xfd, 0x48, 0x6f, 0x37, 0x42, 0xb7, 0xfb, 0x92, 0xd6, 0x19,
    0xda, 0xc2, 0x56, 0x77, 0x29, 0x7e, 0x14, 0x23, 0x24, 0x02, 0xc9, 0x8f, 0x66, 0xf1, 0x28, 0x34,
    0x76, 0x9a, 0x82, 0x89, 0x8d, 0x53, 0xd4, 0x7d, 0xe2, 0xa1, 0xd0, 0xa7, 0x20, 0xfe, 0x9b, 0x49,
    0x2b, 0xea, 0xab, 0xe6, 0xdf, 0x59, 0x1b, 0xe4, 0x35, 0xe8, 0x17, 0xd1, 0x9d, 0xbb, 0x79, 0x33,
    0x94, 0x71, 0x10, 0xd5, 0x09, 0xad, 0x45, 0x9c, 0x38, 0x3d, 0x65, 0xeb, 0xc3, 0x44, 0xf7, 0x3c,
    0x7a, 0xe7, 0x9f, 0x0b, 0x9e, 0xdb, 0xac, 0xc0, 0xfa, 0x7b, 0xa2, 0x80, 0x4b, 0xf9, 0x40, 0x4d,
    0x4e, 0xe5, 0x5a, 0x86, 0xc7, 0x67, 0xc1, 0xe0, 0x74, 0x63, 0x0c, 0xa5, 0x55, 0x5c, 0x03, 0xec,
};

constexpr SBox fallback_sbox_3rd_party_b = {
    0x00, 0xf9, 0x8e, 0xa5, 0x0d, 0x22, 0x24, 0xdc, 0x6e, 0x70, 0x17, 0xd6, 0xa7, 0x59, 0x09, 0x94,
    0xe1, 0xf1, 0xcc, 0x84, 0x3d, 0x99, 0x75, 0xa1, 0x44, 0xbf, 0x9c, 0xfd, 0x37, 0x9a, 0x02, 0x67,
    0x9e, 0x68, 0x33, 0xb7, 0x1a, 0xe5, 0x9d, 0xaf, 0x5a, 0x4b, 0x54, 0x01, 0x45, 0x1e, 0xd3, 0x8d,
    0x7f, 0xc1, 0x5f, 0xa3, 0x87, 0x28, 0x32, 0x30, 0x16, 0x52, 0xd7, 0xc5, 0x69, 0xb4, 0xb8, 0xf0,
    0xab, 0x86, 0xa0, 0x3e, 0x6a, 0x76, 0x63, 0xe6, 0x57, 0x88, 0x07, 0x19, 0x34, 0x48, 0xfa, 0xc6,
    0x64, 0x2f, 0x4d, 0x35, 0x6d, 0xb0, 0x90, 0x13, 0x2d, 0x55, 0xbe, 0xe3, 0x74, 0x61, 0x7a, 0x40,
    0xbc, 0x95, 0x05, 0xb3, 0x96, 0xc7, 0x25, 0x2e, 0x21, 0x8b, 0x2a, 0x79, 0xae, 0xeb, 0xf3, 0x10,
    0xd5, 0x39, 0x4f, 0xca, 0xd8, 0x0f, 0x60, 0xde, 0x62, 0x51, 0xcf, 0x65, 0x1f, 0x0e, 0x23, 0x78,
    0x31, 0xc8, 0xc4, 0xad, 0xf4, 0xda, 0xba, 0xc0, 0xed, 0xf2, 0xf8, 0xa8, 0x3f, 0x36, 0x9b, 0x8c,
    0x20, 0x3b, 0x6b, 0xa2, 0x9f, 0xff, 0x11, 0x29, 0x27, 0xf6, 0x04, 0x71, 0x7d, 0x03, 0x85, 0x06,
    0x7e, 0xdd, 0xea, 0xfe, 0x83, 0x6f, 0xe4, 0x73, 0xb6, 0x98, 0x08, 0xee, 0x8f, 0x15, 0x56, 0x1b,
    0x38, 0x41, 0x4a, 0xfb, 0xdb, 0x14, 0x4c, 0xe0, 0x26, 0xd4, 0xb9, 0x43, 0xcb, 0xaa, 0x42, 0x49,
    0x80, 0x5e, 0x72, 0x3a, 0xd9, 0xe2, 0x2c, 0xc2, 0xec, 0x8a, 0xe9, 0x53, 0x18, 0x3c, 0x47, 0xd2,
    0xef, 0x93, 0x92, 0xf7, 0x0c, 0xbb, 0x97, 0xd0, 0x5b, 0x81, 0xc3, 0xbd, 0x58, 0x12, 0xdf, 0xe8,
    0x5c, 0xac, 0xf5, 0x2b, 0x89, 0xb1, 0xc9, 0x50, 0xd1, 0x1d, 0x4e, 0x46, 0x0b, 0xfc, 0x7c, 0x77,
    0x1c, 0xa4, 0x66, 0xb5, 0x6c, 0x82, 0xce, 0xe7, 0xb2, 0xa6, 0x5d, 0x7b, 0x91, 0xcd, 0xa9, 0x0a,
};

constexpr SBox keygen_sbox_1st_party = {
    0x70, 0x51, 0x03, 0x86, 0x40, 0x0d, 0x4f, 0xeb, 0x3e, 0xcc, 0xd1, 0x87, 0x35, 0xbd, 0xf5, 0x0b,
    0x5e, 0xd0, 0xf8, 0xf2, 0xd5, 0xe2, 0x6c, 0x31, 0x0c, 0xad, 0xfc, 0x21, 0xc3, 0x78, 0xc1, 0x06,
    0xc2, 0x4c, 0x55, 0xe6, 0x4a, 0x34, 0x48, 0x11, 0x1e, 0xda, 0xe7, 0x1a, 0x84, 0xa0, 0x96, 0xa7,
    0xe3, 0x7f, 0xaf, 0x63, 0x9c, 0xfa, 0x23, 0x5b, 0x79, 0xc8, 0x9e, 0xba, 0xb2, 0xc9, 0x22, 0x12,
    0x4b, 0xb3, 0xa1, 0xb6, 0x32, 0x49, 0xa2, 0xe1, 0x89, 0x39, 0x10, 0x66, 0xc5, 0x07, 0x8f, 0x54,
    0xea, 0x91, 0xca, 0x3f, 0xf9, 0x19, 0xf0, 0xd7, 0x46, 0xbc, 0x28, 0x1b, 0x61, 0xe8, 0x2f, 0x6a,
    0xae, 0x9d, 0xf6, 0x4e, 0x09, 0x14, 0x77, 0x4d, 0xdb, 0x1f, 0x2e, 0x7b, 0x7c, 0xf1, 0x43, 0xa3,
    0x00, 0xb8, 0x13, 0x8c, 0x85, 0xb9, 0x29, 0x75, 0x88, 0xfd, 0xd2, 0x56, 0x1c, 0x50, 0x97, 0x41,
    0xe5, 0x3b, 0x60, 0xb5, 0xc0, 0x64, 0xee, 0x98, 0xd6, 0x2d, 0x25, 0xa4, 0xaa, 0xcd, 0x7d, 0xa8,
    0x83, 0xc6, 0xab, 0xbe, 0x44, 0x99, 0x26, 0x3c, 0xce, 0x9f, 0xbf, 0xd3, 0xcb, 0x76, 0x7a, 0x7e,
    0x82, 0x01, 0x8a, 0x9a, 0x80, 0x1d, 0x0e, 0xb0, 0x5c, 0xd4, 0x38, 0x62, 0xf4, 0x30, 0xe0, 0x8e,
    0x53, 0xb7, 0x02, 0x57, 0xac, 0xa6, 0x52, 0x0a, 0x6d, 0x92, 0x65, 0x17, 0x24, 0x33, 0x45, 0x72,
    0x74, 0xb1, 0xb4, 0xf7, 0x5d, 0xed, 0x2c, 0xff, 0x47, 0x37, 0x5a, 0x90, 0xbb, 0xdf, 0x2a, 0x16,
    0x59, 0x95, 0xd9, 0xc4, 0x27, 0x67, 0x73, 0xc7, 0x68, 0xfe, 0xa5, 0xdd, 0x6b, 0x5f, 0x93, 0xd8,
    0xec, 0x05, 0x3a, 0x8d, 0x6e, 0xfb, 0x3d, 0xa9, 0x69, 0x36, 0xf3, 0x94, 0xde, 0xef, 0x15, 0x6f,
    0x8b, 0x9b, 0x08, 0x0f, 0xdc, 0x81, 0x18, 0x20, 0x04, 0xe4, 0x71, 0xcf, 0xe9, 0x2b, 0x42, 0x58,
};

constexpr std::array<SBox, 8> sboxes_1st_party = {{
    {
        0x01, 0xa0, 0xa9, 0x62, 0xd6, 0x3f, 0x85, 0xa7, 0xb6, 0xd4, 0xfa, 0x15, 0x66, 0x17, 0x09,
        0xbd, 0x5d, 0x14, 0x34, 0x26, 0x59, 0x72, 0x91, 0x54, 0x06, 0x4f, 0xf8, 0xb0, 0x5b, 0x74,
        0x93, 0x99, 0x8c, 0xf2, 0x45, 0xcd, 0xea, 0x4e, 0xad, 0x10, 0x4a, 0xe5, 0xca, 0xee, 0xdf,
        0xc6, 0x6f, 0x9f, 0x88, 0x8e, 0x02, 0xcc, 0x08, 0xa8, 0x77, 0x94, 0x6d, 0x21, 0xb1, 0x28,
        0xe4, 0x39, 0x79, 0x96, 0x60, 0x71, 0x81, 0x16, 0x2e, 0xe6, 0x78, 0xb9, 0xc4, 0x46, 0x9a,
        0x42, 0xae, 0xb7, 0x7c, 0x43, 0xb3, 0x22, 0x1a, 0x86, 0xc2, 0x32, 0x3d, 0x2d, 0x9c, 0xd2,
        0x29, 0xe9, 0x63, 0x9b, 0xd1, 0x31, 0x38, 0x5e, 0x1e, 0x36, 0x41, 0xbb, 0x03, 0x18, 0x2b,
        0x3e, 0xbf, 0x68, 0x61, 0xfc, 0x52, 0xc0, 0xde, 0xe0, 0x0a, 0x58, 0x13, 0x5a, 0x00, 0xbe,
        0x1c, 0x90, 0x0e, 0x53, 0x12, 0xfd, 0xe2, 0x6e, 0xba, 0xce, 0x24, 0x27, 0x44, 0x7f, 0x87,
        0xa3, 0xa1, 0xd5, 0x50, 0x40, 0xe3, 0xf9, 0x83, 0xf7, 0xc7, 0xa2, 0x35, 0xc8, 0xdb, 0x19,
        0xab, 0x2f, 0x11, 0x25, 0xed, 0x33, 0x9e, 0x55, 0xe1, 0x48, 0xaf, 0x73, 0x84, 0xda, 0x2a,
        0xaa, 0x51, 0xeb, 0x9d, 0x95, 0xb2, 0xcb, 0xe7, 0x70, 0x80, 0xfe, 0x4c, 0x65, 0x04, 0xef,
        0xc5, 0xf1, 0xc3, 0x3a, 0xb4, 0xf5, 0x5f, 0x23, 0x89, 0xdd, 0x30, 0xa5, 0x8b, 0xd3, 0xf6,
        0xdc, 0x4d, 0x64, 0xd7, 0xf0, 0x8f, 0xec, 0x56, 0x37, 0x5c, 0xa4, 0x0d, 0x07, 0x76, 0x8a,
        0x2c, 0x0b, 0xb5, 0xd8, 0xc1, 0x1f, 0xe8, 0x3b, 0xf4, 0x4b, 0x1b, 0x47, 0x6c, 0x49, 0x67,
        0x7b, 0x92, 0xcf, 0x75, 0x7e, 0x20, 0xd9, 0x7d, 0x3c, 0x97, 0x7a, 0xd0, 0x05, 0x6b, 0x0f,
        0x1d, 0xfb, 0x82, 0x98, 0x57, 0x8d, 0xf3, 0x6a, 0xbc, 0xac, 0xc9, 0xa6, 0xff, 0xb8, 0x69,
        0x0c,
    },
    {
        0x4c, 0x4d, 0x72, 0x07, 0x5a, 0x49, 0x33, 0x8d, 0xa2, 0xab, 0x46, 0x3d, 0x63, 0x0d, 0xa0,
        0x97, 0xff, 0xf0, 0xf5, 0xfa, 0xc0, 0xe9, 0xdb, 0x62, 0xe4, 0xe1, 0x74, 0x43, 0xdc, 0x86,
        0x18, 0x29, 0x37, 0xf4, 0x06, 0xe2, 0xed, 0x6f, 0x90, 0x48, 0x1e, 0x2d, 0x1d, 0xea, 0x73,
        0x94, 0x54, 0xdf, 0x25, 0xf6, 0x47, 0x27, 0xd9, 0x11, 0x77, 0xc9, 0x84, 0x1c, 0x5b, 0x5c,
        0x51, 0x81, 0xa6, 0x22, 0x3e, 0x24, 0x96, 0xc8, 0x8a, 0xec, 0x82, 0x7c, 0x09, 0xb8, 0x45,
        0x4a, 0x57, 0xbb, 0x2f, 0x50, 0x75, 0x8e, 0x61, 0x70, 0x8c, 0x6c, 0xaf, 0xd0, 0xfd, 0xb4,
        0x1b, 0xae, 0xde, 0xfe, 0x3b, 0xb5, 0x36, 0xbd, 0x55, 0x01, 0x0e, 0x9c, 0x41, 0x56, 0x5f,
        0xb3, 0x26, 0x03, 0x83, 0xba, 0x13, 0x4b, 0xca, 0xc5, 0x0a, 0xf8, 0x60, 0xa5, 0xb9, 0xc7,
        0xc3, 0x98, 0x32, 0xfb, 0x12, 0xf9, 0xa7, 0x92, 0xaa, 0x68, 0xf3, 0x78, 0x7e, 0x05, 0x20,
        0x21, 0x02, 0xe8, 0xbf, 0xf2, 0xb0, 0x59, 0x8f, 0xd2, 0xcb, 0x87, 0x65, 0x15, 0xf1, 0x1a,
        0xb2, 0x30, 0xad, 0xee, 0x58, 0xa3, 0x8b, 0x66, 0x1f, 0x2c, 0xd7, 0x5d, 0x19, 0x85, 0xa8,
        0xe6, 0xd3, 0x6b, 0xa1, 0x0c, 0x91, 0x93, 0x6a, 0x5e, 0x0b, 0x79, 0xe3, 0xdd, 0x00, 0x4f,
        0x3c, 0x89, 0x6e, 0x71, 0x69, 0xa9, 0xac, 0x40, 0xe5, 0x99, 0x28, 0xc6, 0x31, 0x4e, 0x7a,
        0xcd, 0x08, 0x9e, 0x7d, 0xef, 0x17, 0xfc, 0x88, 0xd8, 0xa4, 0x6d, 0x44, 0x95, 0xd1, 0xb7,
        0xd4, 0x9b, 0xbe, 0x2a, 0x34, 0x64, 0x2b, 0xcf, 0x2e, 0xeb, 0x38, 0xce, 0x23, 0xe0, 0x3a,
        0x3f, 0xf7, 0x7b, 0x9f, 0x10, 0x53, 0xbc, 0x52, 0x67, 0x16, 0xe7, 0x80, 0x76, 0x04, 0xc4,
        0xb6, 0xc1, 0xc2, 0x7f, 0x9a, 0xda, 0xd5, 0x39, 0x42, 0x14, 0x9d, 0xb1, 0x0f, 0x35, 0xd6,
        0xcc,
    },
    {
        0xb9, 0xda, 0x38, 0x0c, 0xa2, 0x9c, 0x09, 0x1f, 0x06, 0xb1, 0xb6, 0xfd, 0x1a, 0x69, 0x23,
        0x30, 0xc4, 0xde, 0x01, 0xd1, 0xf4, 0x58, 0x29, 0x37, 0x1c, 0x7d, 0xd5, 0xbf, 0xff, 0xbd,
        0xc8, 0xc9, 0xcf, 0x65, 0xbe, 0x7b, 0x78, 0x97, 0x98, 0x67, 0x08, 0xb3, 0x26, 0x57, 0xf7,
        0xfa, 0x40, 0xad, 0x8e, 0x75, 0xa6, 0x7c, 0xdb, 0x91, 0x8b, 0x51, 0x99, 0xd4, 0x17, 0x7a,
        0x90, 0x8d, 0xce, 0x63, 0xcb, 0x4e, 0xa0, 0xab, 0x18, 0x3a, 0x5b, 0x50, 0x7f, 0x21, 0x74,
        0xc1, 0xbb, 0xb8, 0xb7, 0xba, 0x0b, 0x35, 0x95, 0x31, 0x59, 0x9a, 0x4d, 0x04, 0x07, 0x1e,
        0x5a, 0x76, 0x13, 0xf3, 0x71, 0x83, 0xf0, 0x86, 0x03, 0xa8, 0x39, 0x42, 0xaa, 0x28, 0xe6,
        0xe4, 0xd8, 0x5d, 0xd3, 0xd0, 0x6e, 0x6f, 0x96, 0xfb, 0x5e, 0xbc, 0x56, 0xc2, 0x5f, 0x85,
        0x9b, 0xe7, 0xaf, 0xd2, 0x3b, 0x84, 0x6a, 0xa7, 0x53, 0xc5, 0x44, 0x49, 0xa5, 0xf9, 0x36,
        0x72, 0x3d, 0x2c, 0xd9, 0x1b, 0xa1, 0xf5, 0x4f, 0x93, 0x9d, 0x68, 0x47, 0x41, 0x16, 0xca,
        0x2a, 0x4c, 0xa3, 0x87, 0xd6, 0xe5, 0x19, 0x2e, 0x77, 0x15, 0x6d, 0x70, 0xc0, 0xdf, 0xb2,
        0x00, 0x46, 0xed, 0xc6, 0x6c, 0x43, 0x60, 0x92, 0x2d, 0xa9, 0x22, 0x45, 0x8f, 0x34, 0x55,
        0xae, 0xa4, 0x0a, 0x66, 0x32, 0xe0, 0xdc, 0x02, 0xac, 0xe8, 0x20, 0x8c, 0x89, 0x62, 0x4a,
        0xfe, 0xee, 0xc3, 0xe3, 0x3c, 0xf1, 0x79, 0x05, 0xe9, 0xf6, 0x27, 0x33, 0xcc, 0xf2, 0x9e,
        0x11, 0x81, 0x7e, 0x80, 0x10, 0x8a, 0x82, 0x9f, 0x48, 0x0d, 0xd7, 0xb4, 0xfc, 0x2f, 0xb5,
        0xc7, 0xdd, 0x88, 0x14, 0x6b, 0x2b, 0x54, 0xea, 0x1d, 0x94, 0x5c, 0xb0, 0xef, 0x12, 0x24,
        0xcd, 0xeb, 0xe1, 0xe2, 0x64, 0x73, 0x3f, 0x0e, 0x52, 0x61, 0x25, 0x3e, 0xf8, 0x0f, 0x4b,
        0xec,
    },
    {
        0xc0, 0x00, 0x30, 0xf6, 0x02, 0x49, 0x3d, 0x10, 0x6e, 0x20, 0xc9, 0xa6, 0x2f, 0xfe, 0x2c,
        0x2b, 0x75, 0x2e, 0x45, 0x26, 0xab, 0x48, 0xa9, 0x80, 0xfc, 0x04, 0xcc, 0xd3, 0xb5, 0xba,
        0xa3, 0x38, 0x31, 0x7d, 0x01, 0xd9, 0xa7, 0x7b, 0x96, 0xb6, 0x63, 0x69, 0x4e, 0xf7, 0xde,
        0xe0, 0x78, 0xca, 0x50, 0xaa, 0x41, 0x91, 0x65, 0x88, 0xe4, 0x21, 0x85, 0xda, 0x3a, 0x27,
        0xbe, 0x1c, 0x3e, 0x42, 0x5e, 0x17, 0x52, 0x7f, 0x1f, 0x89, 0x24, 0x6f, 0x8f, 0x5c, 0x67,
        0x74, 0x0e, 0x12, 0x87, 0x8d, 0xe9, 0x34, 0xed, 0x73, 0xc4, 0xf8, 0x61, 0x5b, 0x05, 0xdf,
        0x59, 0x4c, 0x97, 0x79, 0x83, 0x18, 0xa4, 0x55, 0x95, 0xeb, 0xbd, 0x53, 0xf5, 0xf1, 0x57,
        0x66, 0x46, 0x9f, 0xb2, 0x81, 0x09, 0x51, 0x86, 0x22, 0x16, 0xdd, 0x23, 0x93, 0x76, 0x29,
        0xc2, 0xd7, 0x1d, 0xd4, 0xbf, 0x36, 0x3f, 0xea, 0x4b, 0x11, 0x32, 0xb9, 0x62, 0x54, 0x60,
        0xd6, 0x6d, 0x43, 0x9a, 0x0d, 0x92, 0x9c, 0xb0, 0xef, 0x58, 0x6c, 0x9d, 0x77, 0x2d, 0x70,
        0xfa, 0xf3, 0xb3, 0x0b, 0xe2, 0x40, 0x7e, 0xf4, 0x8a, 0xe5, 0x8c, 0x3c, 0x56, 0x71, 0xd1,
        0x64, 0xe1, 0x82, 0x0a, 0xcb, 0x13, 0x15, 0x90, 0xec, 0x03, 0x99, 0xaf, 0x14, 0x5d, 0x0f,
        0x33, 0x4a, 0x94, 0xa5, 0xa8, 0x35, 0x1b, 0xe3, 0x6a, 0xc6, 0x28, 0xff, 0x4d, 0xe7, 0x25,
        0x84, 0xac, 0x08, 0xae, 0xc5, 0xa2, 0x2a, 0xb8, 0x37, 0x0c, 0x7a, 0xa0, 0xc3, 0xce, 0xad,
        0x06, 0x1a, 0x9e, 0x8b, 0xfb, 0xd5, 0xf0, 0xc1, 0x1e, 0xd0, 0xb4, 0x9b, 0xb1, 0x44, 0xf2,
        0x47, 0xc7, 0x68, 0xcf, 0x72, 0xbb, 0x4f, 0x5a, 0xf9, 0xdc, 0x6b, 0xdb, 0xd2, 0xe8, 0x7c,
        0xc8, 0xee, 0x98, 0xa1, 0xe6, 0xd8, 0x39, 0x07, 0x5f, 0xfd, 0x8e, 0x19, 0xb7, 0x3b, 0xbc,
        0xcd,
    },
    {
        0x7c, 0xe3, 0x81, 0x73, 0xb2, 0x11, 0xbf, 0x6f, 0x20, 0x98, 0xfe, 0x75, 0x96, 0xef, 0x6c,
        0xda, 0x50, 0xe1, 0x09, 0x72, 0x54, 0x45, 0xba, 0x34, 0x80, 0x5b, 0xed, 0x3e, 0x53, 0x2c,
        0x87, 0xa4, 0x57, 0xf3, 0x33, 0x3f, 0x3c, 0xb7, 0x67, 0xb4, 0xa3, 0x25, 0x60, 0x4f, 0x07,
        0x6b, 0x1b, 0x47, 0x15, 0x0f, 0xe4, 0x0a, 0xea, 0xd1, 0x32, 0x78, 0x36, 0x49, 0x8d, 0x4b,
        0xd2, 0xbc, 0xa5, 0xdc, 0x1d, 0x0d, 0x4d, 0xcd, 0x9a, 0x82, 0x5f, 0xfc, 0x94, 0x65, 0xbe,
        0xe2, 0xf4, 0xc9, 0x1e, 0x44, 0xcb, 0x9e, 0x0c, 0x64, 0x71, 0x26, 0x63, 0xb3, 0x14, 0xe8,
        0x40, 0x70, 0x8a, 0x0e, 0x19, 0x42, 0x6d, 0xac, 0x88, 0x10, 0x5c, 0xdf, 0x41, 0xa9, 0xad,
        0xe5, 0xfb, 0x74, 0xcc, 0xd5, 0x06, 0x8e, 0x59, 0x86, 0xce, 0x1f, 0x3d, 0x76, 0xe0, 0x8f,
        0xb9, 0x77, 0x27, 0x7b, 0xa6, 0xd8, 0x29, 0xd3, 0xec, 0xb8, 0x13, 0xf7, 0xfa, 0xc3, 0x51,
        0x6a, 0xde, 0x4a, 0x5a, 0xeb, 0xc2, 0x8b, 0x23, 0x48, 0x92, 0xcf, 0x62, 0xa8, 0x99, 0xf8,
        0xd0, 0x2e, 0x85, 0x61, 0x43, 0xc8, 0xbd, 0xf0, 0x05, 0x93, 0xca, 0x4e, 0xf1, 0x7d, 0x30,
        0xfd, 0xc4, 0x69, 0x66, 0x2f, 0x08, 0xb1, 0x52, 0xf9, 0x21, 0xe6, 0x7a, 0x2b, 0xdd, 0x39,
        0x84, 0xff, 0xc0, 0x91, 0xd6, 0x37, 0xd4, 0x7f, 0x2d, 0x9b, 0x5d, 0xa1, 0x3b, 0x6e, 0xb5,
        0xc5, 0x46, 0x04, 0xf5, 0x90, 0xee, 0x7e, 0x83, 0x1c, 0x03, 0x56, 0xb6, 0xaa, 0x00, 0x17,
        0x01, 0x35, 0x55, 0x79, 0x0b, 0x12, 0xbb, 0x1a, 0x31, 0xe7, 0x02, 0x28, 0x16, 0xc1, 0xf6,
        0xa2, 0xdb, 0x18, 0x9c, 0x89, 0x68, 0x38, 0x97, 0xab, 0xc7, 0x2a, 0xd7, 0x3a, 0xf2, 0xc6,
        0x24, 0x4c, 0xb0, 0x58, 0xa0, 0x22, 0x5e, 0x9d, 0xd9, 0xa7, 0xe9, 0xae, 0xaf, 0x8c, 0x95,
        0x9f,
    },
    {
        0x28, 0xb7, 0x20, 0xd7, 0xb0, 0x30, 0xc3, 0x09, 0x19, 0xc0, 0x67, 0xd6, 0x00, 0x3c, 0x7e,
        0xe7, 0xe9, 0xf4, 0x08, 0x5a, 0xf8, 0xb8, 0x2e, 0x05, 0xa6, 0x25, 0x9e, 0x5c, 0xd8, 0x15,
        0x0d, 0xe1, 0xf6, 0x11, 0x54, 0x6b, 0xcd, 0x21, 0x46, 0x66, 0x5e, 0x84, 0xad, 0x06, 0x38,
        0x29, 0x44, 0xc5, 0xa2, 0xce, 0xf1, 0xaa, 0xc1, 0x40, 0x71, 0x86, 0xb5, 0xef, 0xfc, 0x36,
        0xa8, 0xcb, 0x0a, 0x48, 0x27, 0x45, 0x64, 0xa3, 0xaf, 0x8c, 0xb2, 0xc6, 0x9f, 0x07, 0x89,
        0xdc, 0x17, 0xd3, 0x49, 0x79, 0xfb, 0xfe, 0x1d, 0xd0, 0xb9, 0x88, 0x43, 0x52, 0xbc, 0x01,
        0x78, 0x2b, 0x7d, 0x94, 0xc7, 0x0e, 0xde, 0xa5, 0xd5, 0x9b, 0xcc, 0xf7, 0x61, 0x7a, 0xc2,
        0x74, 0x81, 0x39, 0x03, 0xab, 0x96, 0xa0, 0x37, 0xbd, 0x2d, 0x72, 0x75, 0x3f, 0xc9, 0xd4,
        0x8e, 0x6f, 0xf9, 0x8d, 0xed, 0x62, 0xdb, 0x1c, 0xdf, 0x04, 0xac, 0x1b, 0x6c, 0x14, 0x4b,
        0x63, 0xf0, 0xbf, 0xb4, 0x82, 0xec, 0x7b, 0x1a, 0x59, 0x92, 0xd2, 0x10, 0x60, 0xb6, 0x3d,
        0x5f, 0xe6, 0x80, 0x6e, 0x70, 0xc4, 0xf2, 0x35, 0xd9, 0x7c, 0xee, 0xe5, 0x41, 0xa4, 0x5b,
        0x50, 0xdd, 0xbb, 0x4c, 0xf3, 0x1f, 0x9d, 0x5d, 0x57, 0x55, 0x51, 0x97, 0xe3, 0x58, 0x42,
        0x4d, 0x9c, 0x73, 0xba, 0xc8, 0x77, 0x31, 0x69, 0x26, 0xae, 0xea, 0x8a, 0xda, 0x22, 0xb3,
        0x87, 0x56, 0xfa, 0x93, 0x0b, 0x34, 0x16, 0x33, 0xe8, 0xe4, 0x53, 0xbe, 0xa9, 0xb1, 0x3a,
        0x3e, 0xf5, 0x90, 0x6a, 0xcf, 0x3b, 0x12, 0xfd, 0x8f, 0x9a, 0xa7, 0x47, 0x91, 0x99, 0xeb,
        0x0f, 0x24, 0xff, 0x23, 0x18, 0x85, 0x4e, 0x7f, 0x0c, 0xe0, 0xa1, 0xe2, 0xd1, 0x2c, 0x2a,
        0x4a, 0x02, 0x4f, 0x1e, 0x95, 0x68, 0x8b, 0x98, 0x83, 0x6d, 0x76, 0xca, 0x65, 0x32, 0x13,
        0x2f,
    },
    {
        0xc3, 0x82, 0x9a, 0xa4, 0xba, 0x81, 0x60, 0x37, 0x34, 0x35, 0xfc, 0x80, 0xa8, 0x51, 0x65,
        0x67, 0xed, 0x30, 0x5f, 0x10, 0xd3, 0x4a, 0x27, 0x2f, 0x13, 0xb9, 0x2a, 0xd2, 0xcc, 0xe1,
        0xef, 0xae, 0xeb, 0xbe, 0xf4, 0xbd, 0xcf, 0x43, 0xb3, 0xc5, 0x88, 0x84, 0xb7, 0xdd, 0x39,
        0x40, 0xce, 0x48, 0x6d, 0x9b, 0x72, 0x61, 0x7e, 0xe7, 0xa1, 0x4e, 0x53, 0x2e, 0x77, 0x3b,
        0xe2, 0xc9, 0x36, 0x22, 0x1b, 0x6e, 0x73, 0xb1, 0x03, 0xb2, 0x4c, 0x87, 0xa9, 0xd4, 0x4d,
        0x0f, 0xd8, 0x15, 0x6c, 0xaa, 0x18, 0xf6, 0x49, 0x57, 0x5d, 0xfb, 0x7a, 0x14, 0x94, 0x63,
        0xa0, 0x11, 0xb0, 0x9e, 0xde, 0x05, 0x46, 0xc8, 0xee, 0x47, 0xdb, 0xdc, 0x24, 0x89, 0x9c,
        0x91, 0x97, 0x29, 0xe9, 0x7b, 0xc1, 0x07, 0x1e, 0xb8, 0xfd, 0xfe, 0xac, 0xc6, 0x62, 0x98,
        0x4f, 0xf1, 0x79, 0xe0, 0xe8, 0x6b, 0x78, 0x56, 0xb6, 0x8d, 0x04, 0x50, 0x86, 0xca, 0x6f,
        0x20, 0xe6, 0xea, 0xe5, 0x76, 0x17, 0x1c, 0x74, 0x7f, 0xbc, 0x0d, 0x2c, 0x85, 0xf7, 0x66,
        0x96, 0xe4, 0x8b, 0x75, 0x3f, 0x4b, 0xd9, 0x38, 0xaf, 0x7c, 0xda, 0x0b, 0x83, 0x2d, 0x31,
        0x32, 0xa2, 0xf5, 0x1d, 0x59, 0x41, 0x45, 0xbf, 0x3c, 0x1f, 0xf8, 0xf9, 0x8a, 0xd0, 0x16,
        0x25, 0x69, 0x12, 0x99, 0x9d, 0x21, 0x95, 0xab, 0x01, 0xa6, 0xd7, 0xb5, 0xc0, 0x7d, 0xff,
        0x58, 0x0e, 0x3a, 0x92, 0xd1, 0x55, 0xe3, 0x08, 0x9f, 0xd6, 0x3e, 0x52, 0x8e, 0xfa, 0xa3,
        0xc7, 0x02, 0xcd, 0xdf, 0x8f, 0x64, 0x19, 0x8c, 0xf3, 0xa7, 0x0c, 0x5e, 0x0a, 0x6a, 0x09,
        0xf0, 0x93, 0x5b, 0x42, 0xc2, 0x06, 0x23, 0xec, 0x71, 0xad, 0xb4, 0xcb, 0xbb, 0x70, 0x28,
        0xd5, 0x1a, 0x5c, 0x33, 0x68, 0x5a, 0x00, 0x44, 0x90, 0xa5, 0xc4, 0x26, 0x3d, 0x2b, 0xf2,
        0x54,
    },
    {
        0x96, 0xad, 0xda, 0x1f, 0xed, 0x33, 0xe1, 0x81, 0x69, 0x08, 0x0d, 0x0a, 0xdb, 0x35, 0x77,
        0x9a, 0x64, 0xd1, 0xfc, 0x78, 0xaa, 0x1b, 0xd0, 0x67, 0xa0, 0xdd, 0xfa, 0x6c, 0x63, 0x71,
        0x05, 0x84, 0x17, 0x6a, 0x89, 0x4f, 0x66, 0x7f, 0xc6, 0x50, 0x55, 0x92, 0x6f, 0xbd, 0xe7,
        0xd2, 0x40, 0x72, 0x8d, 0xbb, 0xec, 0x06, 0x42, 0x8a, 0xe4, 0x88, 0x9d, 0x7e, 0x7a, 0x82,
        0x27, 0x13, 0x41, 0x1a, 0xaf, 0xc8, 0xa4, 0x76, 0xb4, 0xc2, 0xfe, 0x6d, 0x1c, 0xd9, 0x61,
        0x30, 0xb3, 0x7c, 0xea, 0xf7, 0x29, 0x0f, 0xf2, 0x3b, 0x51, 0xc1, 0xde, 0x5f, 0xe5, 0x2a,
        0x2f, 0x99, 0x0b, 0x5d, 0xa3, 0x2b, 0x4a, 0xab, 0x95, 0xa5, 0xd3, 0x58, 0x56, 0xee, 0x28,
        0x31, 0x00, 0xcc, 0x15, 0x46, 0xca, 0xe6, 0x86, 0x38, 0x3c, 0x65, 0xf5, 0xe3, 0x9f, 0xd6,
        0x5b, 0x09, 0x49, 0x83, 0x70, 0x2d, 0x53, 0xa9, 0x7d, 0xe2, 0xc4, 0xac, 0x8e, 0x5e, 0xb8,
        0x25, 0xf4, 0xb9, 0x57, 0xf3, 0xf1, 0x68, 0x47, 0xb2, 0xa2, 0x59, 0x20, 0xce, 0x34, 0x79,
        0x5c, 0x90, 0x0e, 0x1e, 0xbe, 0xd5, 0x22, 0x23, 0xb1, 0xc9, 0x18, 0x62, 0x16, 0x2e, 0x91,
        0x3e, 0x07, 0x8f, 0xd8, 0x3f, 0x93, 0x3d, 0xd4, 0x9b, 0xdf, 0x85, 0x21, 0xfb, 0x11, 0x74,
        0x97, 0xc7, 0xd7, 0xdc, 0x4c, 0x19, 0x45, 0x98, 0xe9, 0x43, 0x02, 0x4b, 0xbc, 0xc3, 0x04,
        0x9c, 0x6b, 0xf0, 0x75, 0x52, 0xa7, 0x26, 0xf6, 0xc5, 0xba, 0xcf, 0xb0, 0xb7, 0xae, 0x5a,
        0xa1, 0xbf, 0x03, 0x8b, 0x80, 0x12, 0x6e, 0x0c, 0xeb, 0xf9, 0xc0, 0x44, 0x24, 0xef, 0x10,
        0xf8, 0xa8, 0x8c, 0xe8, 0x7b, 0xff, 0x9e, 0x2c, 0xcd, 0x60, 0x36, 0x87, 0xb5, 0x94, 0xa6,
        0x54, 0x73, 0x3a, 0x14, 0x4e, 0x01, 0x1d, 0xb6, 0xfd, 0x37, 0x48, 0x4d, 0x39, 0xcb, 0xe0,
        0x32,
    },
}};

constexpr SBox keygen_sbox_3rd_party = {
    0xa4, 0x38, 0x33, 0x69, 0xd4, 0xf5, 0x6a, 0xa8, 0xd7, 0xec, 0x74, 0xa6, 0x7b, 0xc7, 0x3c, 0x41,
    0xb7, 0x21, 0x98, 0x4a, 0x8d, 0xc8, 0x28, 0xf3, 0xf8, 0x32, 0x91, 0xab, 0xad, 0x34, 0x86, 0x30,
    0x4c, 0xbc, 0xff, 0xce, 0x14, 0x1e, 0x2e, 0xf9, 0x31, 0x59, 0x85, 0x87, 0xfc, 0xc2, 0xfd, 0x3e,
    0xbb, 0x56, 0x88, 0x8b, 0xbf, 0x47, 0x61, 0x92, 0xb1, 0x94, 0x7a, 0x51, 0x9c, 0x8f, 0xb0, 0x49,
    0x9b, 0xb9, 0x05, 0x36, 0x10, 0x3b, 0xd5, 0xd3, 0x0c, 0x96, 0xcb, 0x79, 0x60, 0x4e, 0xa2, 0xb5,
    0xd1, 0xee, 0x17, 0xba, 0x1f, 0x09, 0x78, 0x93, 0xb4, 0x35, 0x3a, 0x9d, 0x1d, 0x39, 0xc9, 0x3f,
    0x7c, 0x2f, 0xca, 0x0b, 0x37, 0xe6, 0x1b, 0xd0, 0x67, 0xd9, 0x18, 0x84, 0xe1, 0x42, 0x5a, 0x7e,
    0xb8, 0x97, 0xdf, 0x54, 0x70, 0x03, 0x6e, 0xdb, 0x63, 0xae, 0x55, 0x5d, 0xc4, 0x4b, 0x89, 0xcc,
    0x13, 0xb3, 0x90, 0xf6, 0xe9, 0xb2, 0xa3, 0xbe, 0x2d, 0x19, 0x73, 0x23, 0x20, 0x7f, 0x40, 0x02,
    0x9e, 0xc3, 0x6b, 0x1a, 0x52, 0xda, 0x29, 0x80, 0x0d, 0x5f, 0xd6, 0x8e, 0x0e, 0x22, 0xe3, 0x6d,
    0xdc, 0x58, 0x75, 0xe0, 0x01, 0x0a, 0x26, 0x50, 0x08, 0xa9, 0xfe, 0x4d, 0x15, 0xf7, 0x6c, 0xcf,
    0x4f, 0xf0, 0x64, 0x5e, 0xc6, 0x68, 0x8a, 0xac, 0xde, 0xb6, 0x48, 0x66, 0xed, 0x6f, 0x71, 0x45,
    0x3d, 0xeb, 0x9f, 0x04, 0x25, 0x5c, 0xbd, 0x65, 0xdd, 0x53, 0xcd, 0x24, 0xef, 0x72, 0xe8, 0x9a,
    0xe7, 0x16, 0xc1, 0x00, 0x77, 0x2c, 0x7d, 0xa5, 0x2a, 0x44, 0x62, 0x27, 0xf4, 0xc0, 0x07, 0x1c,
    0x5b, 0x99, 0x82, 0x57, 0x06, 0xfa, 0xfb, 0x12, 0xaf, 0xa7, 0xd2, 0xaa, 0x2b, 0x8c, 0xf2, 0xd8,
    0x83, 0xa0, 0x76, 0x11, 0xc5, 0x43, 0x81, 0x46, 0xe5, 0xf1, 0xea, 0x95, 0xe2, 0xe4, 0x0f, 0xa1,
};

constexpr std::array<SBox, 8> sboxes_3rd_party = {{
    {
        0x2b, 0xd2, 0xa5, 0x8e, 0x26, 0x09, 0x0f, 0xf7, 0x45, 0x5b, 0x3c, 0xfd, 0x8c, 0x72, 0x22,
        0xbf, 0xca, 0xda, 0xe7, 0xaf, 0x16, 0xb2, 0x5e, 0x8a, 0x6f, 0x94, 0xb7, 0xd6, 0x1c, 0xb1,
        0x29, 0x4c, 0xb5, 0x43, 0x18, 0x9c, 0x31, 0xce, 0xb6, 0x84, 0x71, 0x60, 0x7f, 0x2a, 0x6e,
        0x35, 0xf8, 0xa6, 0x54, 0xea, 0x74, 0x88, 0xac, 0x03, 0x19, 0x1b, 0x3d, 0x79, 0xfc, 0xee,
        0x42, 0x9f, 0x93, 0xdb, 0x80, 0xad, 0x8b, 0x15, 0x41, 0x5d, 0x48, 0xcd, 0x7c, 0xa3, 0x2c,
        0x32, 0x1f, 0x63, 0xd1, 0xed, 0x4f, 0x04, 0x66, 0x1e, 0x46, 0x9b, 0xbb, 0x38, 0x06, 0x7e,
        0x95, 0xc8, 0x5f, 0x4a, 0x51, 0x6b, 0x97, 0xbe, 0x2e, 0x98, 0xbd, 0xec, 0x0e, 0x05, 0x0a,
        0xa0, 0x01, 0x52, 0x85, 0xc0, 0xd8, 0x3b, 0xfe, 0x12, 0x64, 0xe1, 0xf3, 0x24, 0x4b, 0xf5,
        0x49, 0x7a, 0xe4, 0x4e, 0x34, 0x25, 0x08, 0x53, 0x1a, 0xe3, 0xef, 0x86, 0xdf, 0xf1, 0x91,
        0xeb, 0xc6, 0xd9, 0xd3, 0x83, 0x14, 0x1d, 0xb0, 0xa7, 0x0b, 0x10, 0x40, 0x89, 0xb4, 0xd4,
        0x3a, 0x02, 0x0c, 0xdd, 0x2f, 0x5a, 0x56, 0x28, 0xae, 0x2d, 0x55, 0xf6, 0xc1, 0xd5, 0xa8,
        0x44, 0xcf, 0x58, 0x9d, 0xb3, 0x23, 0xc5, 0xa4, 0x3e, 0x7d, 0x30, 0x13, 0x6a, 0x61, 0xd0,
        0xf0, 0x3f, 0x67, 0xcb, 0x0d, 0xff, 0x92, 0x68, 0xe0, 0x81, 0x69, 0x62, 0xab, 0x75, 0x59,
        0x11, 0xf2, 0xc9, 0x07, 0xe9, 0xc7, 0xa1, 0xc2, 0x78, 0x33, 0x17, 0x6c, 0xf9, 0xc4, 0xb8,
        0xb9, 0xdc, 0x27, 0x90, 0xbc, 0xfb, 0x70, 0xaa, 0xe8, 0x96, 0x73, 0x39, 0xf4, 0xc3, 0x77,
        0x87, 0xde, 0x00, 0xa2, 0x9a, 0xe2, 0x7b, 0xfa, 0x36, 0x65, 0x6d, 0x20, 0xd7, 0x57, 0x5c,
        0x37, 0x8f, 0x4d, 0x9e, 0x47, 0xa9, 0xe5, 0xcc, 0x99, 0x8d, 0x76, 0x50, 0xba, 0xe6, 0x82,
        0x21,
    },
    {
        0xb0, 0x0d, 0xeb, 0x75, 0x6d, 0x22, 0xce, 0xe7, 0x63, 0x2c, 0xde, 0xe2, 0xf5, 0x45, 0xba,
        0x44, 0x95, 0x1f, 0xe5, 0x39, 0xd7, 0xa5, 0xb5, 0x8d, 0x07, 0x93, 0xfd, 0xbc, 0x62, 0x7e,
        0x89, 0xb3, 0x48, 0x59, 0x7f, 0xd3, 0xbe, 0xd0, 0xf2, 0x36, 0x53, 0x9f, 0xe6, 0xb1, 0x82,
        0xc0, 0x94, 0xb8, 0x2f, 0x03, 0x24, 0x3c, 0x00, 0x0a, 0x42, 0x65, 0xe8, 0x5c, 0x96, 0xd5,
        0xa9, 0x90, 0xb9, 0x85, 0xe9, 0xea, 0x0e, 0x16, 0x7b, 0x84, 0x29, 0x4e, 0x5e, 0x87, 0x9d,
        0x35, 0x06, 0x86, 0xe1, 0xe0, 0xe3, 0xec, 0xef, 0xc8, 0x52, 0x5f, 0x64, 0xe4, 0x46, 0x30,
        0x5a, 0xcd, 0xed, 0xd1, 0x51, 0x81, 0x0c, 0x3a, 0x66, 0xfe, 0x04, 0x67, 0x1e, 0xd6, 0xf7,
        0x8c, 0xbd, 0x33, 0xb2, 0x91, 0xd4, 0xa8, 0x4a, 0x14, 0x0b, 0x79, 0x58, 0x88, 0xb6, 0xc1,
        0x19, 0x56, 0xf8, 0x27, 0xac, 0x23, 0x99, 0xb4, 0xc3, 0x12, 0xf3, 0x20, 0x78, 0x34, 0x49,
        0xdc, 0xd8, 0x05, 0xbb, 0x55, 0x8a, 0x71, 0x11, 0x21, 0xc7, 0xdb, 0xf1, 0x9b, 0xf6, 0x9e,
        0x17, 0xf9, 0xfa, 0xfb, 0xa7, 0x60, 0x8e, 0x6e, 0x8b, 0xca, 0x15, 0xab, 0xdd, 0x3d, 0xa6,
        0xc6, 0x80, 0xc2, 0xda, 0x4c, 0x1d, 0x4f, 0x5d, 0x92, 0x69, 0x98, 0xa2, 0x32, 0x9c, 0x2b,
        0x6a, 0x41, 0x1a, 0x6c, 0xa3, 0xc5, 0x72, 0xaa, 0xcc, 0x6b, 0x10, 0x5b, 0x4d, 0x57, 0x28,
        0x68, 0xee, 0x09, 0x40, 0xf0, 0xad, 0x26, 0x2e, 0x2d, 0x9a, 0xa0, 0x7a, 0xcb, 0x74, 0x3e,
        0x1b, 0x70, 0xfc, 0x43, 0xc4, 0x77, 0x08, 0xb7, 0xcf, 0x1c, 0x7d, 0x18, 0x0f, 0x13, 0x25,
        0xbf, 0x61, 0x01, 0x37, 0xae, 0xd2, 0x2a, 0x7c, 0x97, 0xaf, 0x3f, 0x50, 0xdf, 0x6f, 0xa1,
        0xa4, 0xff, 0x38, 0xc9, 0x83, 0x8f, 0x54, 0x73, 0x02, 0x47, 0xd9, 0x31, 0x76, 0x3b, 0x4b,
        0xf4,
    },
    {
        0x9c, 0x49, 0xc7, 0xb6, 0x14, 0x21, 0x5f, 0x52, 0xf3, 0xb7, 0xd5, 0xf2, 0x95, 0xa0, 0x25,
        0x54, 0xd9, 0xe7, 0xb0, 0xbf, 0x69, 0x13, 0x78, 0xf0, 0x33, 0xe0, 0x8c, 0x94, 0xdb, 0xe6,
        0xa7, 0xb5, 0xec, 0xc2, 0x63, 0xe3, 0x18, 0xb9, 0x4b, 0x74, 0xe5, 0x01, 0x8e, 0xb1, 0x27,
        0x35, 0xb4, 0x3d, 0x26, 0x48, 0x1f, 0xfa, 0xbb, 0x90, 0xd3, 0x98, 0xdf, 0xa4, 0x37, 0x67,
        0x5d, 0x80, 0xd2, 0x62, 0xe1, 0x6f, 0x96, 0x2d, 0xa8, 0x6a, 0x97, 0x30, 0x89, 0xea, 0x3a,
        0xbe, 0x22, 0x19, 0x41, 0xac, 0x79, 0xf1, 0x85, 0xee, 0x70, 0xc0, 0x0a, 0x1b, 0x7f, 0xc1,
        0xd0, 0xa6, 0xf4, 0x16, 0xfc, 0xba, 0x4e, 0x4a, 0xc9, 0x12, 0x8a, 0x1a, 0x9d, 0xc5, 0x38,
        0xd4, 0xde, 0xad, 0xc4, 0x43, 0x03, 0xb3, 0xf7, 0xa9, 0x59, 0x6d, 0xa3, 0x20, 0x36, 0x4d,
        0x86, 0x5b, 0xae, 0x61, 0x17, 0xa5, 0x93, 0xdc, 0x50, 0x2c, 0xef, 0x2a, 0x68, 0x75, 0x7a,
        0x8b, 0xd6, 0xfb, 0x00, 0x9e, 0x2e, 0x02, 0xcd, 0xed, 0x47, 0xab, 0x82, 0xbd, 0x71, 0x11,
        0xb2, 0x66, 0x5e, 0x3c, 0x7c, 0x08, 0x6e, 0xf6, 0x09, 0x45, 0x3f, 0x5a, 0x0f, 0x84, 0xcc,
        0xdd, 0x73, 0x0d, 0xf8, 0x92, 0xe8, 0x0b, 0xe9, 0xcf, 0x4f, 0x1c, 0xda, 0x10, 0x0e, 0x9f,
        0xa2, 0x58, 0xff, 0x40, 0x87, 0x77, 0x2f, 0x60, 0x4c, 0xc3, 0x28, 0x88, 0x15, 0x81, 0x06,
        0x56, 0xd1, 0x55, 0x8d, 0x9b, 0x57, 0x83, 0x65, 0x91, 0x34, 0xce, 0x53, 0x42, 0x04, 0x0c,
        0x46, 0x7d, 0x07, 0x29, 0xfe, 0x24, 0x2b, 0x6c, 0x05, 0x32, 0xcb, 0x72, 0x1e, 0x7e, 0xaa,
        0xe4, 0x99, 0x6b, 0xe2, 0xa1, 0xd8, 0x1d, 0x3e, 0xc6, 0xc8, 0x5c, 0x8f, 0xd7, 0x64, 0x39,
        0xb8, 0x44, 0x23, 0x31, 0x7b, 0x51, 0xca, 0x3b, 0xf5, 0xf9, 0xfd, 0x76, 0xeb, 0x9a, 0xaf,
        0xbc,
    },
    {
        0x94, 0xc5, 0x26, 0x3f, 0xf6, 0xf8, 0x93, 0x6e, 0x73, 0x84, 0x82, 0xe7, 0x54, 0x2c, 0x57,
        0xaa, 0x46, 0x6d, 0x7a, 0x7e, 0xf7, 0xe6, 0x9a, 0x9f, 0xd6, 0xed, 0xe1, 0xf0, 0xf5, 0x67,
        0xde, 0xf4, 0xd1, 0x41, 0xfc, 0x4a, 0x53, 0xc8, 0x4d, 0x30, 0x4e, 0x36, 0x40, 0x31, 0x8d,
        0x98, 0x0c, 0xa2, 0xc0, 0xea, 0x6c, 0x0a, 0x18, 0x58, 0x8f, 0xee, 0x00, 0x62, 0x83, 0xe2,
        0x0d, 0x5b, 0x8b, 0xc7, 0xd7, 0x04, 0x90, 0xa1, 0xe9, 0xcd, 0x16, 0x15, 0x56, 0x78, 0x92,
        0x77, 0xd5, 0x12, 0xe3, 0xce, 0x19, 0xbe, 0x34, 0x50, 0x51, 0x4b, 0x32, 0xc1, 0x06, 0x07,
        0x37, 0x35, 0x60, 0xac, 0xad, 0x2b, 0x27, 0xeb, 0x08, 0x9b, 0xdb, 0x7b, 0x4f, 0x03, 0x79,
        0x11, 0x17, 0xd3, 0xf1, 0xd9, 0xa6, 0x66, 0xf3, 0xcf, 0x72, 0x99, 0x3a, 0xca, 0x3d, 0x0b,
        0x89, 0x47, 0x24, 0xdf, 0x65, 0x14, 0x38, 0x61, 0xc2, 0x6f, 0x13, 0xaf, 0x91, 0x3e, 0x2a,
        0x33, 0xda, 0xe0, 0xa5, 0x3b, 0x1b, 0x1d, 0xa0, 0x22, 0x6a, 0xc4, 0x95, 0xfa, 0xb6, 0xb2,
        0x69, 0x9c, 0xf2, 0x80, 0x44, 0x21, 0xef, 0x9d, 0xa7, 0xcb, 0xa9, 0x96, 0x1e, 0x43, 0x76,
        0x8a, 0x7f, 0x3c, 0xd0, 0xa8, 0x59, 0x20, 0xdc, 0x81, 0xc9, 0xcc, 0x97, 0xa4, 0xbc, 0xbd,
        0x7d, 0xab, 0xa3, 0xb4, 0x52, 0xe4, 0xb0, 0x8c, 0xb9, 0x2f, 0x9e, 0xb7, 0x4c, 0xe8, 0x45,
        0x63, 0x09, 0x64, 0x5d, 0x88, 0xd2, 0x1c, 0x74, 0x87, 0x48, 0x5e, 0x55, 0x6b, 0xb3, 0xec,
        0x39, 0x05, 0x2d, 0x28, 0x10, 0xb1, 0xc6, 0xfe, 0x70, 0x29, 0x5a, 0xfb, 0x49, 0x23, 0xdd,
        0x86, 0x01, 0xe5, 0xd4, 0x25, 0x42, 0xae, 0x85, 0x75, 0xff, 0x68, 0x2e, 0x5c, 0x1f, 0x02,
        0xba, 0xb5, 0xbf, 0x7c, 0xfd, 0x71, 0xf9, 0x8e, 0x5f, 0xb8, 0x0e, 0x0f, 0xc3, 0x1a, 0xbb,
        0xd8,
    },
    {
        0xaa, 0xf3, 0xc7, 0xa9, 0x43, 0x20, 0xc6, 0xd0, 0xf4, 0x07, 0x00, 0x9f, 0x89, 0x8e, 0xba,
        0xe0, 0x5a, 0x24, 0x1f, 0x2a, 0xdd, 0xd4, 0xe3, 0x0d, 0xb0, 0xed, 0xbe, 0xb2, 0x2b, 0x75,
        0x2e, 0x66, 0x17, 0x6e, 0x7c, 0x52, 0x76, 0x91, 0x01, 0x59, 0x61, 0x3c, 0x86, 0x2c, 0xfd,
        0x2d, 0xf6, 0x0b, 0xab, 0xe5, 0x22, 0x41, 0x67, 0x72, 0xc2, 0xad, 0xbd, 0xfe, 0x96, 0xfa,
        0x7b, 0x9e, 0x40, 0x5f, 0x71, 0x98, 0x4c, 0x88, 0xfb, 0xeb, 0x6b, 0xc9, 0xce, 0xbb, 0xc0,
        0x03, 0x5b, 0x3a, 0xae, 0x15, 0x3e, 0x56, 0x4d, 0xd8, 0x3b, 0x50, 0x35, 0x84, 0x97, 0x34,
        0xc4, 0x54, 0xcc, 0x4e, 0x5e, 0xcd, 0xe6, 0x68, 0x4f, 0xb3, 0xb4, 0x9d, 0x5d, 0xa5, 0xa3,
        0x78, 0xe9, 0x38, 0x7e, 0x08, 0x60, 0x6c, 0xdb, 0x7f, 0xbf, 0x82, 0x30, 0xe2, 0x48, 0x19,
        0x6d, 0x1a, 0xa2, 0xdf, 0x58, 0x95, 0x69, 0x51, 0x31, 0xda, 0x0c, 0x4b, 0xa4, 0x0a, 0x45,
        0x27, 0xd7, 0x1e, 0x65, 0x87, 0x1b, 0xb7, 0xb5, 0x42, 0x8c, 0x26, 0x6f, 0x25, 0xea, 0xfc,
        0xef, 0x8a, 0xa7, 0x10, 0xbc, 0xf5, 0xa0, 0x21, 0x44, 0x57, 0x02, 0xd2, 0x2f, 0x63, 0x3d,
        0xd5, 0x79, 0x9c, 0x05, 0xf2, 0xe4, 0xcf, 0x29, 0xf1, 0x53, 0x0e, 0x7a, 0xb1, 0x14, 0x12,
        0xf9, 0xf7, 0x11, 0x8b, 0x36, 0xa6, 0xb6, 0xd6, 0xa1, 0x18, 0x39, 0xec, 0x04, 0xc1, 0xe8,
        0x8d, 0x28, 0x99, 0x73, 0x90, 0x70, 0xd3, 0x1d, 0x8f, 0xcb, 0xf8, 0xf0, 0xb9, 0xde, 0x77,
        0x92, 0x32, 0xe7, 0xee, 0x85, 0x33, 0x13, 0xa8, 0x9a, 0x3f, 0xc8, 0xe1, 0x80, 0x9b, 0x06,
        0x23, 0x37, 0x09, 0xd1, 0xff, 0xac, 0x81, 0x94, 0x64, 0xaf, 0x4a, 0x46, 0x7d, 0x16, 0xc3,
        0x55, 0x47, 0xca, 0xd9, 0x0f, 0x1c, 0x6a, 0x74, 0xdc, 0x49, 0x83, 0xc5, 0xb8, 0x5c, 0x93,
        0x62,
    },
    {
        0xdc, 0x2f, 0xd4, 0xf0, 0x86, 0x18, 0xb1, 0x8c, 0x4f, 0xb9, 0xd5, 0x97, 0x1b, 0x43, 0x4c,
        0x1d, 0xea, 0x5b, 0xc3, 0x11, 0x82, 0xfc, 0xa2, 0x4a, 0xa0, 0x7f, 0x47, 0xaf, 0xa3, 0x0a,
        0x59, 0xad, 0xe5, 0x06, 0x2a, 0x4e, 0x6e, 0xe3, 0x40, 0xdb, 0x53, 0x83, 0x49, 0xb4, 0xba,
        0x71, 0x95, 0x9b, 0x51, 0xb3, 0xac, 0xd1, 0xae, 0xc4, 0xcf, 0x4b, 0x4d, 0x69, 0x16, 0xec,
        0x0b, 0xa5, 0x91, 0x73, 0x07, 0x3b, 0x19, 0x84, 0xc2, 0x6d, 0x8d, 0xb2, 0xbd, 0x25, 0x48,
        0x96, 0xc5, 0x20, 0x1a, 0xa1, 0x8b, 0xc6, 0xe1, 0x3f, 0x28, 0xa7, 0xe7, 0x12, 0x76, 0xb0,
        0x1c, 0x8e, 0xd0, 0x45, 0x90, 0xca, 0x98, 0xd6, 0xa6, 0x36, 0x32, 0x72, 0x8a, 0xa8, 0x0d,
        0x3a, 0x9f, 0x37, 0x3c, 0x89, 0xdd, 0x67, 0x22, 0x81, 0xe9, 0x15, 0x41, 0x42, 0xf5, 0x2e,
        0x66, 0xd2, 0x61, 0x80, 0x0f, 0x54, 0x93, 0xaa, 0xed, 0x5d, 0xe4, 0x7d, 0x30, 0xc9, 0x3d,
        0xe0, 0xe2, 0x26, 0xf2, 0x08, 0xa9, 0x63, 0x24, 0x2d, 0x8f, 0x79, 0x9c, 0xda, 0xbf, 0x55,
        0x62, 0x5c, 0x00, 0xb6, 0x27, 0x6a, 0x0e, 0x92, 0x01, 0xd7, 0xff, 0xeb, 0x1f, 0x74, 0x14,
        0x99, 0xf7, 0x44, 0x87, 0x33, 0x6c, 0x94, 0x13, 0x50, 0x75, 0x68, 0x10, 0x17, 0xbb, 0x2c,
        0x6b, 0x04, 0xdf, 0xbc, 0x35, 0x6f, 0xb7, 0xf1, 0x57, 0xc1, 0xe6, 0x7c, 0xb8, 0x77, 0x65,
        0x3e, 0x5f, 0xd3, 0x64, 0xcb, 0x02, 0x1e, 0x38, 0xab, 0x88, 0x60, 0x5e, 0x9d, 0xbe, 0xc8,
        0xa4, 0xd8, 0x52, 0x7b, 0x29, 0xf4, 0x05, 0x70, 0xce, 0x78, 0x85, 0x39, 0x23, 0xf3, 0x34,
        0x58, 0x9e, 0xfb, 0x31, 0xcc, 0x46, 0xef, 0x9a, 0x5a, 0xfe, 0x7a, 0x56, 0xfd, 0xee, 0xde,
        0x0c, 0xf9, 0xcd, 0xc0, 0x2b, 0xd9, 0xb5, 0x09, 0xf6, 0xc7, 0x21, 0xfa, 0xf8, 0x03, 0xe8,
        0x7e,
    },
    {
        0x9f, 0xbd, 0x79, 0x91, 0xcc, 0x78, 0x7f, 0xd5, 0x1b, 0xee, 0xbb, 0xb7, 0xdc, 0xed, 0x6d,
        0x9b, 0xef, 0x58, 0xfb, 0x5d, 0x67, 0x69, 0x1f, 0x4e, 0xaf, 0x62, 0x4b, 0xb8, 0xa8, 0xae,
        0x08, 0x60, 0x8c, 0xad, 0x24, 0xa1, 0x48, 0x2f, 0x8a, 0x00, 0x90, 0xb1, 0x53, 0x07, 0x26,
        0x12, 0x72, 0x9d, 0x56, 0x1c, 0x51, 0x34, 0x46, 0x2e, 0x81, 0xcf, 0xb3, 0xf3, 0x37, 0xf1,
        0xfa, 0x3c, 0xba, 0x06, 0x54, 0x85, 0x14, 0x59, 0x36, 0x5e, 0xa5, 0x55, 0x0c, 0x52, 0x89,
        0xcb, 0x97, 0xe2, 0x3b, 0x11, 0xbe, 0xd9, 0x1d, 0xfc, 0x13, 0x7d, 0xa0, 0x05, 0x41, 0xc3,
        0xc5, 0xfd, 0x76, 0x33, 0x82, 0xcd, 0x88, 0x16, 0x1a, 0x29, 0x87, 0x1e, 0x40, 0x6f, 0xe8,
        0x03, 0x3a, 0x50, 0xa6, 0xe4, 0xf4, 0x4f, 0xd6, 0x98, 0x39, 0x71, 0xd0, 0xf2, 0xf0, 0x3e,
        0xea, 0x8d, 0x2c, 0xa2, 0xa9, 0x57, 0x0b, 0x8b, 0x0d, 0xaa, 0xe9, 0x31, 0xc2, 0x35, 0x5c,
        0xd1, 0x38, 0xff, 0x01, 0xa7, 0xec, 0x80, 0xf9, 0x23, 0xdf, 0x30, 0xeb, 0x2a, 0xb0, 0x17,
        0xd2, 0x19, 0x6e, 0xda, 0x25, 0x9e, 0xb6, 0x4c, 0xc6, 0x93, 0xdb, 0x75, 0x6b, 0x3f, 0x15,
        0x73, 0xfe, 0x3d, 0x65, 0x66, 0xb5, 0x7a, 0x9a, 0x32, 0xca, 0x68, 0x8e, 0x22, 0xc0, 0x44,
        0x43, 0xa3, 0x0e, 0x21, 0xb9, 0x02, 0x4d, 0xc8, 0xb4, 0x0a, 0xf6, 0xd3, 0x45, 0x96, 0x5a,
        0x84, 0x70, 0x18, 0xd4, 0xbf, 0xc4, 0xa4, 0x9c, 0xc1, 0xe0, 0x47, 0xc7, 0xe5, 0xe6, 0x7e,
        0x63, 0xdd, 0x6a, 0xce, 0x28, 0x94, 0xe7, 0x10, 0xbc, 0x5f, 0x04, 0x64, 0x2d, 0x8f, 0x6c,
        0xf7, 0x2b, 0x92, 0x83, 0xd8, 0x99, 0xe3, 0xde, 0xd7, 0x5b, 0xab, 0x4a, 0x74, 0x77, 0xc9,
        0xf5, 0x7b, 0x0f, 0x20, 0xf8, 0x61, 0x42, 0x86, 0x7c, 0x49, 0x95, 0xac, 0x27, 0xe1, 0x09,
        0xb2,
    },
    {
        0x0b, 0xd3, 0x6f, 0x0a, 0x39, 0x77, 0x21, 0x17, 0x8f, 0xc7, 0xd5, 0xe5, 0xb5, 0xb9, 0x11,
        0xa4, 0xdc, 0xa3, 0xcd, 0xbb, 0x1a, 0x81, 0x8a, 0xb7, 0xd9, 0x5b, 0x47, 0x61, 0x8e, 0x88,
        0xb8, 0x0c, 0x3b, 0x3a, 0x9a, 0x54, 0xfd, 0x2d, 0x53, 0xb4, 0xc3, 0x5f, 0x69, 0x60, 0xb1,
        0xab, 0x26, 0x56, 0x83, 0x7e, 0xb3, 0x3d, 0x4a, 0x0d, 0x5a, 0x1d, 0x1e, 0x59, 0xd6, 0x35,
        0x8c, 0xf3, 0x9b, 0x18, 0x66, 0xbf, 0xe2, 0x62, 0x92, 0x7b, 0xff, 0x2e, 0x5c, 0xcf, 0x25,
        0xc4, 0xaf, 0xd2, 0x29, 0x63, 0xa5, 0xb2, 0xf9, 0x6a, 0xc1, 0xfb, 0x14, 0x87, 0x55, 0x73,
        0x2c, 0x44, 0x31, 0x79, 0xf4, 0x2a, 0x41, 0x01, 0x24, 0x80, 0x19, 0x65, 0x4d, 0x9d, 0x9c,
        0xe8, 0x13, 0xa2, 0xc0, 0xc5, 0xfe, 0x6b, 0xce, 0x50, 0xf8, 0x04, 0x48, 0xa8, 0x06, 0x16,
        0xbe, 0x67, 0xba, 0xea, 0xc6, 0x0f, 0xd7, 0x74, 0x85, 0xb6, 0xd8, 0x0e, 0x4c, 0x93, 0x15,
        0xbd, 0xe6, 0x78, 0x03, 0x05, 0x32, 0xf7, 0x34, 0xe4, 0xad, 0x30, 0x98, 0xa1, 0x27, 0x9e,
        0xf6, 0x43, 0x64, 0x3c, 0x49, 0xbc, 0xf0, 0x99, 0xdd, 0x12, 0xd1, 0xc9, 0x5d, 0x7c, 0x22,
        0x75, 0x1f, 0x28, 0x2f, 0x09, 0xc2, 0x84, 0x6d, 0xfa, 0x23, 0x3f, 0x7d, 0x91, 0x89, 0x82,
        0x86, 0x58, 0xdf, 0x76, 0xe9, 0xaa, 0xdb, 0xac, 0x2b, 0xf5, 0x90, 0x42, 0x20, 0xe1, 0xa0,
        0xed, 0xd4, 0x52, 0x10, 0xef, 0x3e, 0xe3, 0x1c, 0xda, 0x96, 0xb0, 0x72, 0x38, 0x9f, 0x7a,
        0x1b, 0xde, 0x02, 0xa6, 0x4e, 0x97, 0x33, 0x36, 0x6e, 0xe0, 0xc8, 0x4f, 0xfc, 0x37, 0x71,
        0xec, 0x94, 0x00, 0x95, 0xd0, 0xa7, 0xcb, 0xf1, 0x70, 0xa9, 0x8b, 0x40, 0xf2, 0x4b, 0x46,
        0x45, 0xee, 0x51, 0x8d, 0xcc, 0x6c, 0xca, 0xeb, 0x7f, 0x68, 0x07, 0xae, 0x5e, 0x57, 0x08,
        0xe7,
    },
}};

}  // namespace

namespace WiimoteEmu
{
void EncryptionKey::GenerateTables(const RandData& rand, const KeyData& key, const SBox& sbox_a,
                                   const SBox& sbox_b)
{
  ft[0] = sbox_a[key[4]] ^ sbox_b[rand[3]];
  ft[1] = sbox_a[key[2]] ^ sbox_b[rand[5]];
  ft[2] = sbox_a[key[5]] ^ sbox_b[rand[7]];
  ft[3] = sbox_a[key[0]] ^ sbox_b[rand[2]];
  ft[4] = sbox_a[key[1]] ^ sbox_b[rand[4]];
  ft[5] = sbox_a[key[3]] ^ sbox_b[rand[9]];
  ft[6] = sbox_a[rand[0]] ^ sbox_b[rand[6]];
  ft[7] = sbox_a[rand[1]] ^ sbox_b[rand[8]];

  sb[0] = sbox_a[key[0]] ^ sbox_b[rand[1]];
  sb[1] = sbox_a[key[5]] ^ sbox_b[rand[4]];
  sb[2] = sbox_a[key[3]] ^ sbox_b[rand[0]];
  sb[3] = sbox_a[key[2]] ^ sbox_b[rand[9]];
  sb[4] = sbox_a[key[4]] ^ sbox_b[rand[7]];
  sb[5] = sbox_a[key[1]] ^ sbox_b[rand[8]];
  sb[6] = sbox_a[rand[3]] ^ sbox_b[rand[5]];
  sb[7] = sbox_a[rand[2]] ^ sbox_b[rand[6]];
}

// Generate key from the 0x40-0x4f data in the extension register.
EncryptionKey KeyGen::GenerateFromExtensionKeyData(const ExtKeyData& ext_key) const
{
  EncryptionKey::RandData rand;
  std::copy_n(ext_key.begin(), rand.size(), rand.rbegin());

  EncryptionKey::KeyData key;
  std::copy_n(ext_key.begin() + rand.size(), key.size(), key.rbegin());

  // The "key" is generated from "rand" and "idx".
  // The tables are generated from "rand" "key" and "idx".
  // But "idx" is not sent to the extension.
  // We brute force which "idx" (0-6) the Wii used to generate its "key".
  for (u32 idx = 0; idx != 7; ++idx)
  {
    if (GenerateKeyData(rand, idx) == key)
      return GenerateTables(rand, key, idx);
  }

  // Retail games never hit this path but some homebrew fills encryption key with 0x00.
  // Real extensions seem to then use entirely differnet "sboxes" for table generation.
  WARN_LOG_FMT(WIIMOTE, "Extension key gen did not match any idx. Generating fallback tables.");
  return GenerateFallbackTables(rand, key);
}

EncryptionKey::KeyData KeyGen1stParty::GenerateKeyData(const EncryptionKey::RandData& rand,
                                                       const u32 idx) const
{
  constexpr u8 ans_tbl[7][6] = {
      {0xa8, 0x77, 0xa6, 0xe0, 0xf7, 0x43}, {0x5a, 0x35, 0x85, 0xe2, 0x72, 0x97},
      {0x8f, 0xb7, 0x1a, 0x62, 0x87, 0x38}, {0x0d, 0x67, 0xc7, 0xbe, 0x4f, 0x3e},
      {0x20, 0x76, 0x37, 0x8f, 0x68, 0xb7}, {0xa9, 0x26, 0x3f, 0x2b, 0x10, 0xe3},
      {0x30, 0x7e, 0x90, 0x0e, 0x85, 0x0a},
  };

  auto& ans = ans_tbl[idx];

  EncryptionKey::RandData t0;
  for (std::size_t i = 0; i != t0.size(); ++i)
    t0[i] = keygen_sbox_1st_party[rand[i]];

  return {
      u8((std::rotr<u8>(ans[0] ^ t0[5], t0[2]) - t0[9]) ^ t0[4]),
      u8((std::rotr<u8>(ans[1] ^ t0[1], t0[0]) - t0[5]) ^ t0[7]),
      u8((std::rotr<u8>(ans[2] ^ t0[6], t0[8]) - t0[2]) ^ t0[0]),
      u8((std::rotr<u8>(ans[3] ^ t0[4], t0[7]) - t0[3]) ^ t0[2]),
      u8((std::rotr<u8>(ans[4] ^ t0[1], t0[6]) - t0[3]) ^ t0[4]),
      u8((std::rotr<u8>(ans[5] ^ t0[7], t0[8]) - t0[5]) ^ t0[9]),
  };
}

EncryptionKey::KeyData KeyGen3rdParty::GenerateKeyData(const EncryptionKey::RandData& rand,
                                                       const u32 idx) const
{
  constexpr u8 ans_tbl[7][6] = {
      {0x54, 0x29, 0x28, 0x4e, 0x0f, 0xfc}, {0xa4, 0x6c, 0x87, 0x50, 0x49, 0x3c},
      {0xf2, 0xb4, 0xb1, 0xf9, 0xc3, 0x58}, {0x2e, 0x83, 0x5f, 0x71, 0x8a, 0x9b},
      {0x82, 0xca, 0xc2, 0xbe, 0xba, 0x75}, {0x91, 0x60, 0x78, 0xa8, 0x2a, 0x4c},
      {0x25, 0x5b, 0x8b, 0xb0, 0xda, 0xaf},
  };

  auto& ans = ans_tbl[idx];

  EncryptionKey::RandData t0;
  for (std::size_t i = 0; i != t0.size(); ++i)
    t0[i] = keygen_sbox_3rd_party[rand[i]];

  return {
      u8(t0[7] ^ (t0[6] + std::rotl<u8>(ans[0] ^ t0[0], t0[1]))),
      u8(t0[1] ^ (t0[3] + std::rotl<u8>(ans[1] ^ t0[4], t0[2]))),
      u8(t0[5] ^ (t0[4] + std::rotl<u8>(ans[2] ^ t0[2], t0[8]))),
      u8(t0[0] ^ (t0[7] + std::rotl<u8>(ans[3] ^ t0[6], t0[9]))),
      u8(t0[1] ^ (t0[8] + std::rotl<u8>(ans[4] ^ t0[5], t0[4]))),
      u8(t0[5] ^ (t0[8] + std::rotl<u8>(ans[5] ^ t0[9], t0[3]))),
  };
}

EncryptionKey KeyGen1stParty::GenerateTables(const EncryptionKey::RandData& rand,
                                             const EncryptionKey::KeyData& key, u32 idx) const
{
  EncryptionKey result;
  result.GenerateTables(rand, key, sboxes_1st_party[idx], sboxes_1st_party[idx + 1]);
  return result;
}

EncryptionKey KeyGen3rdParty::GenerateTables(const EncryptionKey::RandData& rand,
                                             const EncryptionKey::KeyData& key, u32 idx) const
{
  EncryptionKey result;
  result.GenerateTables(rand, key, sboxes_3rd_party[idx], sboxes_3rd_party[idx + 1]);
  return result;
}

EncryptionKey KeyGen1stParty::GenerateFallbackTables(const EncryptionKey::RandData& rand,
                                                     const EncryptionKey::KeyData& key) const
{
  // Reverse engineering did not produce exact tables, only the XOR effect of each.
  // This is the XOR difference between the tables.
  // This also works with 0x97.
  constexpr u8 fallback_sbox_xor = 0x17;

  EncryptionKey result;
  result.GenerateTables(rand, key, fallback_sbox_1st_party_a, fallback_sbox_1st_party_b);
  for (auto& ft : result.ft)
    ft ^= fallback_sbox_xor;
  for (auto& sb : result.sb)
    sb ^= fallback_sbox_xor;
  return result;
}

EncryptionKey KeyGen3rdParty::GenerateFallbackTables(const EncryptionKey::RandData& rand,
                                                     const EncryptionKey::KeyData& key) const
{
  // Reverse engineering did not produce exact tables, only the XOR effect of each.
  // This is the XOR difference between the tables.
  // This also works with 0x20, 0x60, or 0xa0
  constexpr u8 fallback_sbox_xor = 0xe0;

  EncryptionKey result;
  result.GenerateTables(rand, key, fallback_sbox_3rd_party_a, fallback_sbox_3rd_party_b);
  for (auto& ft : result.ft)
    ft ^= fallback_sbox_xor;
  for (auto& sb : result.sb)
    sb ^= fallback_sbox_xor;
  return result;
}

void EncryptionKey::Encrypt(u8* const data, u32 addr, const u32 len) const
{
  for (u32 i = 0; i != len; ++i, ++addr)
    data[i] = (data[i] - ft[addr % 8]) ^ sb[addr % 8];
}

void EncryptionKey::Decrypt(u8* const data, u32 addr, const u32 len) const
{
  for (u32 i = 0; i != len; ++i, ++addr)
    data[i] = (data[i] ^ sb[addr % 8]) + ft[addr % 8];
}

}  // namespace WiimoteEmu
